<?php
// user/my_records.php
require_once '../config/database.php';
require_once '../config/auth.php';
require_once '../models/Record.php';
require_once '../models/Expense.php';

// Check if user is logged in
redirectIfNotLoggedIn();
if (isAdmin()) {
    header('Location: ../admin/dashboard.php');
    exit();
}

$database = new Database();
$db = $database->getConnection();

// Get today's date
$today = date('Y-m-d');

// Get user's records and expenses for today
$record = new Record($db);
$expense = new Expense($db);

$result = $record->getByUserId($_SESSION['user_id'], $today);
$userRecords = [];
while ($row = mysqli_fetch_assoc($result)) {
    $userRecords[] = $row;
}

$result = $expense->getByUserId($_SESSION['user_id'], $today);
$userExpenses = [];
while ($row = mysqli_fetch_assoc($result)) {
    $userExpenses[] = $row;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My Records - Fomax Logistics</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
    <link href="https://fonts.googleapis.com/icon?family=Material+Icons" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Roboto', sans-serif;
            background: #f5f7fa;
            color: #333;
        }
        
        .header {
            background: white;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            padding: 0 20px;
            position: sticky;
            top: 0;
            z-index: 100;
        }
        
        .header-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
            max-width: 1200px;
            margin: 0 auto;
            height: 70px;
        }
        
        .logo {
            display: flex;
            align-items: center;
        }
        
        .logo h1 {
            color: #2a5298;
            font-size: 22px;
            font-weight: 700;
        }
        
        .user-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: #2a5298;
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 500;
        }
        
        .nav-links {
            background: white;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            padding: 0 20px;
        }
        
        .nav-content {
            display: flex;
            max-width: 1200px;
            margin: 0 auto;
        }
        
        .nav-content a {
            padding: 15px 20px;
            text-decoration: none;
            color: #555;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 8px;
            transition: color 0.3s;
            border-bottom: 3px solid transparent;
        }
        
        .nav-content a:hover, .nav-content a.active {
            color: #2a5298;
            border-bottom: 3px solid #2a5298;
        }
        
        .container {
            max-width: 1200px;
            margin: 30px auto;
            padding: 0 20px;
        }
        
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
        }
        
        .page-title {
            font-size: 24px;
            font-weight: 500;
            color: #2a5298;
        }
        
        .btn {
            background: #2a5298;
            color: white;
            border: none;
            padding: 10px 15px;
            border-radius: 6px;
            font-size: 14px;
            font-weight: 500;
            cursor: pointer;
            transition: background 0.3s;
            text-decoration: none;
            display: inline-block;
        }
        
        .btn:hover {
            background: #1a2a6c;
        }
        
        .btn.outline {
            background: transparent;
            border: 1px solid #2a5298;
            color: #2a5298;
        }
        
        .btn.outline:hover {
            background: rgba(42, 82, 152, 0.1);
        }
        
        .card {
            background: white;
            border-radius: 10px;
            padding: 25px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            margin-bottom: 30px;
        }
        
        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        
        .card-title {
            font-size: 18px;
            font-weight: 500;
            color: #2a5298;
        }
        
        .table-container {
            overflow-x: auto;
            margin-bottom: 30px;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
        }
        
        th, td {
            padding: 15px;
            text-align: left;
            border-bottom: 1px solid #eee;
        }
        
        th {
            background: #f8f9fa;
            font-weight: 500;
            color: #555;
        }
        
        tr:hover {
            background: #f8f9fa;
        }
        
        .tab-container {
            margin-bottom: 20px;
        }
        
        .tabs {
            display: flex;
            border-bottom: 1px solid #ddd;
            margin-bottom: 20px;
        }
        
        .tab {
            padding: 12px 20px;
            cursor: pointer;
            background: #f8f9fa;
            border: 1px solid #ddd;
            border-bottom: none;
            border-radius: 6px 6px 0 0;
            margin-right: 5px;
        }
        
        .tab.active {
            background: white;
            border-bottom: 1px solid white;
            margin-bottom: -1px;
            font-weight: 500;
        }
        
        .tab-content {
            display: none;
        }
        
        .tab-content.active {
            display: block;
        }
        
        @media (max-width: 768px) {
            .header-content {
                flex-direction: column;
                height: auto;
                padding: 15px 0;
            }
            
            .nav-content {
                flex-wrap: wrap;
            }
            
            .nav-content a {
                padding: 12px 15px;
                font-size: 14px;
            }
            
            .page-header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
        }
    </style>
</head>
<body>
    <header class="header">
        <div class="header-content">
            <div class="logo">
                <h1>Fomax Logistics</h1>
            </div>
            <div class="user-info">
                <div class="user-avatar"><?php echo strtoupper(substr($_SESSION['username'], 0, 1)); ?></div>
                <div>
                    <div style="font-weight: 500;"><?php echo htmlspecialchars($_SESSION['username']); ?></div>
                    <div style="font-size: 14px; color: #777;">Staff User</div>
                </div>
            </div>
        </div>
    </header>
    
    <nav class="nav-links">
        <div class="nav-content">
            <a href="dashboard.php">
                <span class="material-icons">dashboard</span>
                Dashboard
            </a>
            <a href="add_record.php">
                <span class="material-icons">add_circle</span>
                Add Record
            </a>
            <a href="add_expense.php">
                <span class="material-icons">add_shopping_cart</span>
                Add Expense
            </a>
            <a href="my_records.php" class="active">
                <span class="material-icons">list</span>
                My Records
            </a>
            <a href="../logout.php">
                <span class="material-icons">exit_to_app</span>
                Logout
            </a>
        </div>
    </nav>
    
    <div class="container">
        <div class="page-header">
            <h1 class="page-title">My Records for <?php echo date('F j, Y'); ?></h1>
        </div>
        
        <div class="tab-container">
            <div class="tabs">
                <div class="tab active" data-tab="records">Records</div>
                <div class="tab" data-tab="expenses">Expenses</div>
            </div>
            
            <div id="records" class="tab-content active">
                <div class="card">
                    <div class="card-header">
                        <div class="card-title">Transaction Records</div>
                    </div>
                    <div class="table-container">
                        <table>
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Product</th>
                                    <th>Quantity</th>
                                    <th>Price (₦)</th>
                                    <th>Charges (₦)</th>
                                    <th>Total (₦)</th>
                                    <th>Rider</th>
                                    <th>Payment Mode</th>
                                    <th>Time</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($userRecords)): ?>
                                    <tr>
                                        <td colspan="9" style="text-align: center;">No records found for today</td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($userRecords as $r): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($r['id']); ?></td>
                                            <td><?php echo htmlspecialchars($r['product_name']); ?></td>
                                            <td><?php echo htmlspecialchars($r['quantity']); ?></td>
                                            <td><?php echo number_format($r['price'], 2); ?></td>
                                            <td><?php echo number_format($r['vendor_charges'], 2); ?></td>
                                            <td><?php echo number_format($r['price'] + $r['vendor_charges'], 2); ?></td>
                                            <td><?php echo htmlspecialchars($r['rider_name']); ?></td>
                                            <td><?php echo htmlspecialchars($r['payment_mode']); ?></td>
                                            <td><?php echo date('g:i A', strtotime($r['entry_date'])); ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            
            <div id="expenses" class="tab-content">
                <div class="card">
                    <div class="card-header">
                        <div class="card-title">Expenses</div>
                    </div>
                    <div class="table-container">
                        <table>
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Description</th>
                                    <th>Amount (₦)</th>
                                    <th>Related Record</th>
                                    <th>Time</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($userExpenses)): ?>
                                    <tr>
                                        <td colspan="5" style="text-align: center;">No expenses found for today</td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($userExpenses as $e): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($e['id']); ?></td>
                                            <td><?php echo htmlspecialchars($e['description']); ?></td>
                                            <td><?php echo number_format($e['amount'], 2); ?></td>
                                            <td>
                                                <?php if ($e['record_id']): ?>
                                                    Record #<?php echo htmlspecialchars($e['record_id']); ?>
                                                <?php else: ?>
                                                    <span style="color: #777;">None</span>
                                                <?php endif; ?>
                                            </td>
                                            <td><?php echo date('g:i A', strtotime($e['entry_date'])); ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script>
        // Tab switching functionality
        document.querySelectorAll('.tab').forEach(tab => {
            tab.addEventListener('click', () => {
                // Remove active class from all tabs and content
                document.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));
                document.querySelectorAll('.tab-content').forEach(c => c.classList.remove('active'));
                
                // Add active class to clicked tab
                tab.classList.add('active');
                
                // Show corresponding content
                const tabId = tab.getAttribute('data-tab');
                document.getElementById(tabId).classList.add('active');
            });
        });
    </script>
</body>
</html>