<?php
// admin/users.php
require_once '../config/database.php';
require_once '../config/auth.php';
require_once '../models/User.php';
require_once '../models/Branch.php';

// Check if user is logged in and is admin
redirectIfNotLoggedIn();
redirectIfNotAdmin();

$database = new Database();
$db = $database->getConnection();
$user = new User($db);
$branch = new Branch($db);

$message = '';
$error = '';

// Get all branches for dropdown
$stmt = $branch->getAll();
$branches = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'create':
                $user->username = trim($_POST['username']);
                $password = $_POST['password'];
                $user->branch_id = $_POST['branch_id'];
                $user->role = $_POST['role'];
                
                if (empty($user->username)) {
                    $error = 'Username is required.';
                } else if (empty($password)) {
                    $error = 'Password is required.';
                } else if (empty($user->branch_id)) {
                    $error = 'Please select a branch.';
                } else if (empty($user->role)) {
                    $error = 'Please select a role.';
                } else {
                    // Check if username already exists
                    if ($user->getByUsername($user->username)) {
                        $error = 'Username already exists.';
                    } else {
                        // Hash password
                        $user->password = password_hash($password, PASSWORD_DEFAULT);
                        
                        if ($user->create()) {
                            $message = 'User created successfully.';
                        } else {
                            $error = 'Failed to create user.';
                        }
                    }
                }
                break;
                
            case 'update':
                $user->id = $_POST['id'];
                $user->username = trim($_POST['username']);
                $password = $_POST['password'];
                $user->branch_id = $_POST['branch_id'];
                $user->role = $_POST['role'];
                
                if (empty($user->username)) {
                    $error = 'Username is required.';
                } else if (empty($user->branch_id)) {
                    $error = 'Please select a branch.';
                } else if (empty($user->role)) {
                    $error = 'Please select a role.';
                } else {
                    // Check if username already exists (for other users)
                    $existingUser = new User($db);
                    if ($existingUser->getByUsername($user->username) && $existingUser->id != $user->id) {
                        $error = 'Username already exists.';
                    } else {
                        // Hash password if provided
                        if (!empty($password)) {
                            $user->password = password_hash($password, PASSWORD_DEFAULT);
                        } else {
                            // Keep existing password
                            $currentUser = new User($db);
                            $currentUser->getById($user->id);
                            $user->password = $currentUser->password;
                        }
                        
                        if ($user->update()) {
                            $message = 'User updated successfully.';
                        } else {
                            $error = 'Failed to update user.';
                        }
                    }
                }
                break;
                
            case 'delete':
                // Prevent deleting own account
                if ($_POST['id'] == $_SESSION['user_id']) {
                    $error = 'You cannot delete your own account.';
                } else {
                    $user->id = $_POST['id'];
                    if ($user->delete()) {
                        $message = 'User deleted successfully.';
                    } else {
                        $error = 'Failed to delete user.';
                    }
                }
                break;
        }
    }
}

// Get all users with branch info
$stmt = $user->getAll();
$users = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Users - Fomax Logistics</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
    <link href="https://fonts.googleapis.com/icon?family=Material+Icons" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Roboto', sans-serif;
            background: #f5f7fa;
            color: #333;
        }
        
        .header {
            background: white;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            padding: 0 20px;
            position: sticky;
            top: 0;
            z-index: 100;
        }
        
        .header-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
            max-width: 1200px;
            margin: 0 auto;
            height: 70px;
        }
        
        .logo {
            display: flex;
            align-items: center;
        }
        
        .logo h1 {
            color: #2a5298;
            font-size: 22px;
            font-weight: 700;
        }
        
        .user-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: #2a5298;
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 500;
        }
        
        .nav-links {
            background: white;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            padding: 0 20px;
        }
        
        .nav-content {
            display: flex;
            max-width: 1200px;
            margin: 0 auto;
        }
        
        .nav-content a {
            padding: 15px 20px;
            text-decoration: none;
            color: #555;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 8px;
            transition: color 0.3s;
            border-bottom: 3px solid transparent;
        }
        
        .nav-content a:hover, .nav-content a.active {
            color: #2a5298;
            border-bottom: 3px solid #2a5298;
        }
        
        .container {
            max-width: 1200px;
            margin: 30px auto;
            padding: 0 20px;
        }
        
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
        }
        
        .page-title {
            font-size: 24px;
            font-weight: 500;
            color: #2a5298;
        }
        
        .btn {
            background: #2a5298;
            color: white;
            border: none;
            padding: 10px 15px;
            border-radius: 6px;
            font-size: 14px;
            font-weight: 500;
            cursor: pointer;
            transition: background 0.3s;
            text-decoration: none;
            display: inline-block;
        }
        
        .btn:hover {
            background: #1a2a6c;
        }
        
        .btn.outline {
            background: transparent;
            border: 1px solid #2a5298;
            color: #2a5298;
        }
        
        .btn.outline:hover {
            background: rgba(42, 82, 152, 0.1);
        }
        
        .btn.danger {
            background: #dc3545;
        }
        
        .btn.danger:hover {
            background: #c82333;
        }
        
        .message {
            padding: 15px;
            border-radius: 6px;
            margin-bottom: 20px;
        }
        
        .message.success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .message.error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .card {
            background: white;
            border-radius: 10px;
            padding: 25px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            margin-bottom: 30px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        label {
            display: block;
            margin-bottom: 8px;
            font-weight: 500;
            color: #555;
        }
        
        input, select {
            width: 100%;
            padding: 12px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 16px;
            transition: border-color 0.3s;
        }
        
        input:focus, select:focus {
            border-color: #2a5298;
            outline: none;
            box-shadow: 0 0 0 2px rgba(42, 82, 152, 0.2);
        }
        
        .form-actions {
            display: flex;
            gap: 10px;
            margin-top: 20px;
        }
        
        .table-container {
            overflow-x: auto;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
        }
        
        th, td {
            padding: 15px;
            text-align: left;
            border-bottom: 1px solid #eee;
        }
        
        th {
            background: #f8f9fa;
            font-weight: 500;
            color: #555;
        }
        
        tr:hover {
            background: #f8f9fa;
        }
        
        .actions {
            display: flex;
            gap: 10px;
        }
        
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            z-index: 1000;
            justify-content: center;
            align-items: center;
        }
        
        .modal-content {
            background: white;
            border-radius: 10px;
            width: 100%;
            max-width: 500px;
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.2);
        }
        
        .modal-header {
            padding: 20px 25px;
            border-bottom: 1px solid #eee;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .modal-title {
            font-size: 18px;
            font-weight: 500;
            color: #2a5298;
        }
        
        .close-modal {
            background: none;
            border: none;
            font-size: 24px;
            cursor: pointer;
            color: #777;
        }
        
        .modal-body {
            padding: 25px;
        }
        
        .modal-footer {
            padding: 20px 25px;
            border-top: 1px solid #eee;
            display: flex;
            justify-content: flex-end;
            gap: 10px;
        }
        
        @media (max-width: 768px) {
            .header-content {
                flex-direction: column;
                height: auto;
                padding: 15px 0;
            }
            
            .nav-content {
                flex-wrap: wrap;
            }
            
            .nav-content a {
                padding: 12px 15px;
                font-size: 14px;
            }
            
            .page-header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
        }
    </style>
</head>
<body>
    <header class="header">
        <div class="header-content">
            <div class="logo">
                <h1>Fomax Logistics</h1>
            </div>
            <div class="user-info">
                <div class="user-avatar"><?php echo strtoupper(substr($_SESSION['username'], 0, 1)); ?></div>
                <div>
                    <div style="font-weight: 500;"><?php echo htmlspecialchars($_SESSION['username']); ?></div>
                    <div style="font-size: 14px; color: #777;">Administrator</div>
                </div>
            </div>
        </div>
    </header>
    
    <nav class="nav-links">
        <div class="nav-content">
            <a href="dashboard.php">
                <span class="material-icons">dashboard</span>
                Dashboard
            </a>
            <a href="branches.php">
                <span class="material-icons">location_city</span>
                Branches
            </a>
            <a href="riders.php">
                <span class="material-icons">local_shipping</span>
                Riders
            </a>
            <a href="users.php" class="active">
                <span class="material-icons">people</span>
                Users
            </a>
            <a href="records.php">
                <span class="material-icons">receipt</span>
                Records
            </a>
            <a href="expenses.php">
                <span class="material-icons">account_balance_wallet</span>
                Expenses
            </a>
            <a href="reports.php">
                <span class="material-icons">assessment</span>
                Reports
            </a>
            <a href="../logout.php">
                <span class="material-icons">exit_to_app</span>
                Logout
            </a>
        </div>
    </nav>
    
    <div class="container">
        <div class="page-header">
            <h1 class="page-title">Manage Users</h1>
            <button class="btn" id="addUserBtn">
                <span class="material-icons" style="font-size: 18px; margin-right: 5px;">add</span>
                Add New User
            </button>
        </div>
        
        <?php if ($message): ?>
            <div class="message success"><?php echo htmlspecialchars($message); ?></div>
        <?php endif; ?>
        
        <?php if ($error): ?>
            <div class="message error"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <div class="card">
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Username</th>
                            <th>Branch</th>
                            <th>Role</th>
                            <th>Created At</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($users)): ?>
                            <tr>
                                <td colspan="6" style="text-align: center;">No users found</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($users as $u): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($u['id']); ?></td>
                                    <td><?php echo htmlspecialchars($u['username']); ?></td>
                                    <td><?php echo htmlspecialchars($u['branch_name']); ?></td>
                                    <td>
                                        <?php if ($u['role'] === 'admin'): ?>
                                            <span style="background: #2a5298; color: white; padding: 4px 8px; border-radius: 4px; font-size: 12px;">Admin</span>
                                        <?php else: ?>
                                            <span style="background: #6c757d; color: white; padding: 4px 8px; border-radius: 4px; font-size: 12px;">User</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo date('M j, Y', strtotime($u['created_at'])); ?></td>
                                    <td class="actions">
                                        <button class="btn outline edit-btn" 
                                            data-id="<?php echo $u['id']; ?>" 
                                            data-username="<?php echo htmlspecialchars($u['username']); ?>" 
                                            data-branch="<?php echo $u['branch_id']; ?>" 
                                            data-role="<?php echo $u['role']; ?>">
                                            <span class="material-icons" style="font-size: 16px;">edit</span>
                                        </button>
                                        <?php if ($u['id'] != $_SESSION['user_id']): ?>
                                            <button class="btn danger delete-btn" 
                                                data-id="<?php echo $u['id']; ?>" 
                                                data-username="<?php echo htmlspecialchars($u['username']); ?>">
                                                <span class="material-icons" style="font-size: 16px;">delete</span>
                                            </button>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    
    <!-- Add/Edit User Modal -->
    <div class="modal" id="userModal">
        <div class="modal-content">
            <div class="modal-header">
                <div class="modal-title" id="modalTitle">Add New User</div>
                <button class="close-modal">&times;</button>
            </div>
            <form method="POST" id="userForm">
                <input type="hidden" name="action" id="formAction" value="create">
                <input type="hidden" name="id" id="userId">
                <div class="modal-body">
                    <div class="form-group">
                        <label for="userName">Username</label>
                        <input type="text" id="userName" name="username" required>
                    </div>
                    <div class="form-group">
                        <label for="userPassword">
                            Password <?php if (isset($_POST['action']) && $_POST['action'] === 'update'): ?>(leave blank to keep current)<?php endif; ?>
                        </label>
                        <input type="password" id="userPassword" name="password">
                    </div>
                    <div class="form-group">
                        <label for="userBranch">Branch</label>
                        <select id="userBranch" name="branch_id" required>
                            <option value="">Select a branch</option>
                            <?php foreach ($branches as $b): ?>
                                <option value="<?php echo $b['id']; ?>"><?php echo htmlspecialchars($b['name']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="userRole">Role</label>
                        <select id="userRole" name="role" required>
                            <option value="">Select a role</option>
                            <option value="user">User</option>
                            <option value="admin">Admin</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn outline closeModal">Cancel</button>
                    <button type="submit" class="btn">Save User</button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Delete Confirmation Modal -->
    <div class="modal" id="deleteModal">
        <div class="modal-content">
            <div class="modal-header">
                <div class="modal-title">Confirm Delete</div>
                <button class="close-modal">&times;</button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to delete the user "<span id="deleteUserName"></span>"? This action cannot be undone.</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn outline closeModal">Cancel</button>
                <form method="POST" style="display: inline;">
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="id" id="deleteUserId">
                    <button type="submit" class="btn danger">Delete User</button>
                </form>
            </div>
        </div>
    </div>
    
    <script>
        // Modal elements
        const userModal = document.getElementById('userModal');
        const deleteModal = document.getElementById('deleteModal');
        const addUserBtn = document.getElementById('addUserBtn');
        const closeModalButtons = document.querySelectorAll('.closeModal, .close-modal');
        
        // Form elements
        const formAction = document.getElementById('formAction');
        const userId = document.getElementById('userId');
        const userName = document.getElementById('userName');
        const userPassword = document.getElementById('userPassword');
        const userBranch = document.getElementById('userBranch');
        const userRole = document.getElementById('userRole');
        const modalTitle = document.getElementById('modalTitle');
        
        // Delete elements
        const deleteUserId = document.getElementById('deleteUserId');
        const deleteUserName = document.getElementById('deleteUserName');
        
        // Event listeners
        addUserBtn.addEventListener('click', () => {
            // Reset form
            formAction.value = 'create';
            userId.value = '';
            userName.value = '';
            userPassword.value = '';
            userBranch.value = '';
            userRole.value = '';
            modalTitle.textContent = 'Add New User';
            userModal.style.display = 'flex';
        });
        
        // Edit buttons
        document.querySelectorAll('.edit-btn').forEach(button => {
            button.addEventListener('click', (e) => {
                const id = e.target.closest('.edit-btn').dataset.id;
                const username = e.target.closest('.edit-btn').dataset.username;
                const branch = e.target.closest('.edit-btn').dataset.branch;
                const role = e.target.closest('.edit-btn').dataset.role;
                
                // Fill form
                formAction.value = 'update';
                userId.value = id;
                userName.value = username;
                userPassword.value = ''; // Clear password field
                userBranch.value = branch;
                userRole.value = role;
                modalTitle.textContent = 'Edit User';
                userModal.style.display = 'flex';
            });
        });
        
        // Delete buttons
        document.querySelectorAll('.delete-btn').forEach(button => {
            button.addEventListener('click', (e) => {
                const id = e.target.closest('.delete-btn').dataset.id;
                const username = e.target.closest('.delete-btn').dataset.username;
                
                // Set delete form values
                deleteUserId.value = id;
                deleteUserName.textContent = username;
                deleteModal.style.display = 'flex';
            });
        });
        
        // Close modals
        closeModalButtons.forEach(button => {
            button.addEventListener('click', () => {
                userModal.style.display = 'none';
                deleteModal.style.display = 'none';
            });
        });
        
        // Close modal when clicking outside
        window.addEventListener('click', (e) => {
            if (e.target === userModal) {
                userModal.style.display = 'none';
            }
            if (e.target === deleteModal) {
                deleteModal.style.display = 'none';
            }
        });
    </script>
</body>
</html>