<?php
// admin/branches.php
require_once '../config/database.php';
require_once '../config/auth.php';
require_once '../models/Branch.php';

// Check if user is logged in and is admin
redirectIfNotLoggedIn();
redirectIfNotAdmin();

$database = new Database();
$db = $database->getConnection();
$branch = new Branch($db);

$message = '';
$error = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'create':
                $branch->name = trim($_POST['name']);
                $branch->location = trim($_POST['location']);
                
                if (empty($branch->name)) {
                    $error = 'Branch name is required.';
                } else {
                    if ($branch->create()) {
                        $message = 'Branch created successfully.';
                    } else {
                        $error = 'Failed to create branch.';
                    }
                }
                break;
                
            case 'update':
                $branch->id = $_POST['id'];
                $branch->name = trim($_POST['name']);
                $branch->location = trim($_POST['location']);
                
                if (empty($branch->name)) {
                    $error = 'Branch name is required.';
                } else {
                    if ($branch->update()) {
                        $message = 'Branch updated successfully.';
                    } else {
                        $error = 'Failed to update branch.';
                    }
                }
                break;
                
            case 'delete':
                $branch->id = $_POST['id'];
                if ($branch->delete()) {
                    $message = 'Branch deleted successfully.';
                } else {
                    $error = 'Failed to delete branch.';
                }
                break;
        }
    }
}

// Get all branches
$stmt = $branch->getAll();
$branches = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Branches - Fomax Logistics</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
    <link href="https://fonts.googleapis.com/icon?family=Material+Icons" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Roboto', sans-serif;
            background: #f5f7fa;
            color: #333;
        }
        
        .header {
            background: white;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            padding: 0 20px;
            position: sticky;
            top: 0;
            z-index: 100;
        }
        
        .header-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
            max-width: 1200px;
            margin: 0 auto;
            height: 70px;
        }
        
        .logo {
            display: flex;
            align-items: center;
        }
        
        .logo h1 {
            color: #2a5298;
            font-size: 22px;
            font-weight: 700;
        }
        
        .user-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: #2a5298;
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 500;
        }
        
        .nav-links {
            background: white;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            padding: 0 20px;
        }
        
        .nav-content {
            display: flex;
            max-width: 1200px;
            margin: 0 auto;
        }
        
        .nav-content a {
            padding: 15px 20px;
            text-decoration: none;
            color: #555;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 8px;
            transition: color 0.3s;
            border-bottom: 3px solid transparent;
        }
        
        .nav-content a:hover, .nav-content a.active {
            color: #2a5298;
            border-bottom: 3px solid #2a5298;
        }
        
        .container {
            max-width: 1200px;
            margin: 30px auto;
            padding: 0 20px;
        }
        
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
        }
        
        .page-title {
            font-size: 24px;
            font-weight: 500;
            color: #2a5298;
        }
        
        .btn {
            background: #2a5298;
            color: white;
            border: none;
            padding: 10px 15px;
            border-radius: 6px;
            font-size: 14px;
            font-weight: 500;
            cursor: pointer;
            transition: background 0.3s;
            text-decoration: none;
            display: inline-block;
        }
        
        .btn:hover {
            background: #1a2a6c;
        }
        
        .btn.outline {
            background: transparent;
            border: 1px solid #2a5298;
            color: #2a5298;
        }
        
        .btn.outline:hover {
            background: rgba(42, 82, 152, 0.1);
        }
        
        .btn.danger {
            background: #dc3545;
        }
        
        .btn.danger:hover {
            background: #c82333;
        }
        
        .message {
            padding: 15px;
            border-radius: 6px;
            margin-bottom: 20px;
        }
        
        .message.success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .message.error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .card {
            background: white;
            border-radius: 10px;
            padding: 25px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            margin-bottom: 30px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        label {
            display: block;
            margin-bottom: 8px;
            font-weight: 500;
            color: #555;
        }
        
        input, textarea {
            width: 100%;
            padding: 12px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 16px;
            transition: border-color 0.3s;
        }
        
        input:focus, textarea:focus {
            border-color: #2a5298;
            outline: none;
            box-shadow: 0 0 0 2px rgba(42, 82, 152, 0.2);
        }
        
        .form-actions {
            display: flex;
            gap: 10px;
            margin-top: 20px;
        }
        
        .table-container {
            overflow-x: auto;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
        }
        
        th, td {
            padding: 15px;
            text-align: left;
            border-bottom: 1px solid #eee;
        }
        
        th {
            background: #f8f9fa;
            font-weight: 500;
            color: #555;
        }
        
        tr:hover {
            background: #f8f9fa;
        }
        
        .actions {
            display: flex;
            gap: 10px;
        }
        
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            z-index: 1000;
            justify-content: center;
            align-items: center;
        }
        
        .modal-content {
            background: white;
            border-radius: 10px;
            width: 100%;
            max-width: 500px;
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.2);
        }
        
        .modal-header {
            padding: 20px 25px;
            border-bottom: 1px solid #eee;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .modal-title {
            font-size: 18px;
            font-weight: 500;
            color: #2a5298;
        }
        
        .close-modal {
            background: none;
            border: none;
            font-size: 24px;
            cursor: pointer;
            color: #777;
        }
        
        .modal-body {
            padding: 25px;
        }
        
        .modal-footer {
            padding: 20px 25px;
            border-top: 1px solid #eee;
            display: flex;
            justify-content: flex-end;
            gap: 10px;
        }
        
        @media (max-width: 768px) {
            .header-content {
                flex-direction: column;
                height: auto;
                padding: 15px 0;
            }
            
            .nav-content {
                flex-wrap: wrap;
            }
            
            .nav-content a {
                padding: 12px 15px;
                font-size: 14px;
            }
            
            .page-header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
        }
    </style>
</head>
<body>
    <header class="header">
        <div class="header-content">
            <div class="logo">
                <h1>Fomax Logistics</h1>
            </div>
            <div class="user-info">
                <div class="user-avatar"><?php echo strtoupper(substr($_SESSION['username'], 0, 1)); ?></div>
                <div>
                    <div style="font-weight: 500;"><?php echo htmlspecialchars($_SESSION['username']); ?></div>
                    <div style="font-size: 14px; color: #777;">Administrator</div>
                </div>
            </div>
        </div>
    </header>
    
    <nav class="nav-links">
        <div class="nav-content">
            <a href="dashboard.php">
                <span class="material-icons">dashboard</span>
                Dashboard
            </a>
            <a href="branches.php" class="active">
                <span class="material-icons">location_city</span>
                Branches
            </a>
            <a href="riders.php">
                <span class="material-icons">local_shipping</span>
                Riders
            </a>
            <a href="users.php">
                <span class="material-icons">people</span>
                Users
            </a>
            <a href="records.php">
                <span class="material-icons">receipt</span>
                Records
            </a>
            <a href="expenses.php">
                <span class="material-icons">account_balance_wallet</span>
                Expenses
            </a>
            <a href="reports.php">
                <span class="material-icons">assessment</span>
                Reports
            </a>
            <a href="../logout.php">
                <span class="material-icons">exit_to_app</span>
                Logout
            </a>
        </div>
    </nav>
    
    <div class="container">
        <div class="page-header">
            <h1 class="page-title">Manage Branches</h1>
            <button class="btn" id="addBranchBtn">
                <span class="material-icons" style="font-size: 18px; margin-right: 5px;">add</span>
                Add New Branch
            </button>
        </div>
        
        <?php if ($message): ?>
            <div class="message success"><?php echo htmlspecialchars($message); ?></div>
        <?php endif; ?>
        
        <?php if ($error): ?>
            <div class="message error"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <div class="card">
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Name</th>
                            <th>Location</th>
                            <th>Created At</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($branches)): ?>
                            <tr>
                                <td colspan="5" style="text-align: center;">No branches found</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($branches as $b): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($b['id']); ?></td>
                                    <td><?php echo htmlspecialchars($b['name']); ?></td>
                                    <td><?php echo htmlspecialchars($b['location']); ?></td>
                                    <td><?php echo date('M j, Y', strtotime($b['created_at'])); ?></td>
                                    <td class="actions">
                                        <button class="btn outline edit-btn" data-id="<?php echo $b['id']; ?>" data-name="<?php echo htmlspecialchars($b['name']); ?>" data-location="<?php echo htmlspecialchars($b['location']); ?>">
                                            <span class="material-icons" style="font-size: 16px;">edit</span>
                                        </button>
                                        <button class="btn danger delete-btn" data-id="<?php echo $b['id']; ?>" data-name="<?php echo htmlspecialchars($b['name']); ?>">
                                            <span class="material-icons" style="font-size: 16px;">delete</span>
                                        </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    
    <!-- Add/Edit Branch Modal -->
    <div class="modal" id="branchModal">
        <div class="modal-content">
            <div class="modal-header">
                <div class="modal-title" id="modalTitle">Add New Branch</div>
                <button class="close-modal">&times;</button>
            </div>
            <form method="POST" id="branchForm">
                <input type="hidden" name="action" id="formAction" value="create">
                <input type="hidden" name="id" id="branchId">
                <div class="modal-body">
                    <div class="form-group">
                        <label for="branchName">Branch Name</label>
                        <input type="text" id="branchName" name="name" required>
                    </div>
                    <div class="form-group">
                        <label for="branchLocation">Location</label>
                        <textarea id="branchLocation" name="location" rows="3"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn outline closeModal">Cancel</button>
                    <button type="submit" class="btn">Save Branch</button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Delete Confirmation Modal -->
    <div class="modal" id="deleteModal">
        <div class="modal-content">
            <div class="modal-header">
                <div class="modal-title">Confirm Delete</div>
                <button class="close-modal">&times;</button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to delete the branch "<span id="deleteBranchName"></span>"? This action cannot be undone.</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn outline closeModal">Cancel</button>
                <form method="POST" style="display: inline;">
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="id" id="deleteBranchId">
                    <button type="submit" class="btn danger">Delete Branch</button>
                </form>
            </div>
        </div>
    </div>
    
    <script>
        // Modal elements
        const branchModal = document.getElementById('branchModal');
        const deleteModal = document.getElementById('deleteModal');
        const addBranchBtn = document.getElementById('addBranchBtn');
        const closeModalButtons = document.querySelectorAll('.closeModal, .close-modal');
        
        // Form elements
        const formAction = document.getElementById('formAction');
        const branchId = document.getElementById('branchId');
        const branchName = document.getElementById('branchName');
        const branchLocation = document.getElementById('branchLocation');
        const modalTitle = document.getElementById('modalTitle');
        
        // Delete elements
        const deleteBranchId = document.getElementById('deleteBranchId');
        const deleteBranchName = document.getElementById('deleteBranchName');
        
        // Event listeners
        addBranchBtn.addEventListener('click', () => {
            // Reset form
            formAction.value = 'create';
            branchId.value = '';
            branchName.value = '';
            branchLocation.value = '';
            modalTitle.textContent = 'Add New Branch';
            branchModal.style.display = 'flex';
        });
        
        // Edit buttons
        document.querySelectorAll('.edit-btn').forEach(button => {
            button.addEventListener('click', (e) => {
                const id = e.target.closest('.edit-btn').dataset.id;
                const name = e.target.closest('.edit-btn').dataset.name;
                const location = e.target.closest('.edit-btn').dataset.location;
                
                // Fill form
                formAction.value = 'update';
                branchId.value = id;
                branchName.value = name;
                branchLocation.value = location;
                modalTitle.textContent = 'Edit Branch';
                branchModal.style.display = 'flex';
            });
        });
        
        // Delete buttons
        document.querySelectorAll('.delete-btn').forEach(button => {
            button.addEventListener('click', (e) => {
                const id = e.target.closest('.delete-btn').dataset.id;
                const name = e.target.closest('.delete-btn').dataset.name;
                
                // Set delete form values
                deleteBranchId.value = id;
                deleteBranchName.textContent = name;
                deleteModal.style.display = 'flex';
            });
        });
        
        // Close modals
        closeModalButtons.forEach(button => {
            button.addEventListener('click', () => {
                branchModal.style.display = 'none';
                deleteModal.style.display = 'none';
            });
        });
        
        // Close modal when clicking outside
        window.addEventListener('click', (e) => {
            if (e.target === branchModal) {
                branchModal.style.display = 'none';
            }
            if (e.target === deleteModal) {
                deleteModal.style.display = 'none';
            }
        });
    </script>
</body>
</html>