<?php
// models/Expense.php
class Expense {
    private $conn;
    private $table_name = "expenses";

    public $id;
    public $description;
    public $amount;
    public $record_id;
    public $user_id;
    public $branch_id;
    public $entry_date;

    public function __construct($db) {
        $this->conn = $db;
    }

    // Get all expenses with related data
    public function getAll() {
        $query = "SELECT e.*, r.product_name as record_product, u.username as user_name, b.name as branch_name 
                  FROM " . $this->table_name . " e
                  LEFT JOIN records r ON e.record_id = r.id
                  LEFT JOIN users u ON e.user_id = u.id
                  LEFT JOIN branches b ON e.branch_id = b.id
                  ORDER BY e.entry_date DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt;
    }

    // Get expenses by user ID
    public function getByUserId($user_id, $date = null) {
        $query = "SELECT e.*, r.product_name as record_product, u.username as user_name, b.name as branch_name 
                  FROM " . $this->table_name . " e
                  LEFT JOIN records r ON e.record_id = r.id
                  LEFT JOIN users u ON e.user_id = u.id
                  LEFT JOIN branches b ON e.branch_id = b.id
                  WHERE e.user_id = :user_id";
        
        if($date) {
            $query .= " AND DATE(e.entry_date) = :date";
        }
        
        $query .= " ORDER BY e.entry_date DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":user_id", $user_id);
        
        if($date) {
            $stmt->bindParam(":date", $date);
        }
        
        $stmt->execute();
        return $stmt;
    }

    // Get expenses by filters
    public function getByFilters($branch_id = null, $start_date = null, $end_date = null) {
        $query = "SELECT e.*, r.product_name as record_product, u.username as user_name, b.name as branch_name 
                  FROM " . $this->table_name . " e
                  LEFT JOIN records r ON e.record_id = r.id
                  LEFT JOIN users u ON e.user_id = u.id
                  LEFT JOIN branches b ON e.branch_id = b.id
                  WHERE 1=1";
        
        if($branch_id) {
            $query .= " AND e.branch_id = :branch_id";
        }
        
        if($start_date) {
            $query .= " AND DATE(e.entry_date) >= :start_date";
        }
        
        if($end_date) {
            $query .= " AND DATE(e.entry_date) <= :end_date";
        }
        
        $query .= " ORDER BY e.entry_date DESC";
        
        $stmt = $this->conn->prepare($query);
        
        if($branch_id) {
            $stmt->bindParam(":branch_id", $branch_id);
        }
        
        if($start_date) {
            $stmt->bindParam(":start_date", $start_date);
        }
        
        if($end_date) {
            $stmt->bindParam(":end_date", $end_date);
        }
        
        $stmt->execute();
        return $stmt;
    }

    // Get expense by ID
    public function getById($id) {
        $query = "SELECT e.*, r.product_name as record_product, u.username as user_name, b.name as branch_name 
                  FROM " . $this->table_name . " e
                  LEFT JOIN records r ON e.record_id = r.id
                  LEFT JOIN users u ON e.user_id = u.id
                  LEFT JOIN branches b ON e.branch_id = b.id
                  WHERE e.id = ? LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $id);
        $stmt->execute();
        
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if($row) {
            $this->id = $row['id'];
            $this->description = $row['description'];
            $this->amount = $row['amount'];
            $this->record_id = $row['record_id'];
            $this->user_id = $row['user_id'];
            $this->branch_id = $row['branch_id'];
            $this->entry_date = $row['entry_date'];
            return $row; // Return the full row with related data
        }
        return false;
    }

    // Create expense
    public function create() {
        $query = "INSERT INTO " . $this->table_name . " 
                  SET description=:description, amount=:amount, record_id=:record_id, 
                      user_id=:user_id, branch_id=:branch_id";
        
        $stmt = $this->conn->prepare($query);
        
        // Sanitize
        $this->description = htmlspecialchars(strip_tags($this->description));
        $this->amount = htmlspecialchars(strip_tags($this->amount));
        $this->record_id = htmlspecialchars(strip_tags($this->record_id));
        $this->user_id = htmlspecialchars(strip_tags($this->user_id));
        $this->branch_id = htmlspecialchars(strip_tags($this->branch_id));
        
        // Bind values
        $stmt->bindParam(":description", $this->description);
        $stmt->bindParam(":amount", $this->amount);
        
        // Handle nullable record_id
        if(empty($this->record_id)) {
            $stmt->bindValue(":record_id", null, PDO::PARAM_NULL);
        } else {
            $stmt->bindParam(":record_id", $this->record_id);
        }
        
        $stmt->bindParam(":user_id", $this->user_id);
        $stmt->bindParam(":branch_id", $this->branch_id);
        
        if($stmt->execute()) {
            return $this->conn->lastInsertId();
        }
        return false;
    }

    // Delete expense
    public function delete() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        
        if($stmt->execute()) {
            return true;
        }
        return false;
    }
    
    // Get total expenses
    public function getTotalExpenses($branch_id = null, $date = null) {
        $query = "SELECT SUM(amount) as total_expenses FROM " . $this->table_name . " WHERE 1=1";
        
        if($branch_id) {
            $query .= " AND branch_id = :branch_id";
        }
        
        if($date) {
            $query .= " AND DATE(entry_date) = :date";
        }
        
        $stmt = $this->conn->prepare($query);
        
        if($branch_id) {
            $stmt->bindParam(":branch_id", $branch_id);
        }
        
        if($date) {
            $stmt->bindParam(":date", $date);
        }
        
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row['total_expenses'] ?? 0;
    }
}
?>