<?php
// models/Record.php
class Record {
    private $conn;
    private $table_name = "records";

    public $id;
    public $rider_id;
    public $product_name;
    public $quantity;
    public $price;
    public $location;
    public $vendor_charges;
    public $payment_mode;
    public $notes;
    public $user_id;
    public $branch_id;
    public $entry_date;

    public function __construct($db) {
        $this->conn = $db;
    }

    // Get all records with related data
    public function getAll($limit = null, $offset = null) {
        $query = "SELECT r.*, rid.name as rider_name, u.username as user_name, b.name as branch_name 
                  FROM " . $this->table_name . " r
                  LEFT JOIN riders rid ON r.rider_id = rid.id
                  LEFT JOIN users u ON r.user_id = u.id
                  LEFT JOIN branches b ON r.branch_id = b.id
                  ORDER BY r.entry_date DESC";
        
        if($limit !== null && $offset !== null) {
            $query .= " LIMIT :offset, :limit";
        }
        
        $stmt = $this->conn->prepare($query);
        
        if($limit !== null && $offset !== null) {
            $stmt->bindParam(":limit", $limit, PDO::PARAM_INT);
            $stmt->bindParam(":offset", $offset, PDO::PARAM_INT);
        }
        
        $stmt->execute();
        return $stmt;
    }

    // Get records by user ID
    public function getByUserId($user_id, $date = null) {
        $query = "SELECT r.*, rid.name as rider_name, u.username as user_name, b.name as branch_name 
                  FROM " . $this->table_name . " r
                  LEFT JOIN riders rid ON r.rider_id = rid.id
                  LEFT JOIN users u ON r.user_id = u.id
                  LEFT JOIN branches b ON r.branch_id = b.id
                  WHERE r.user_id = :user_id";
        
        if($date) {
            $query .= " AND DATE(r.entry_date) = :date";
        }
        
        $query .= " ORDER BY r.entry_date DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":user_id", $user_id);
        
        if($date) {
            $stmt->bindParam(":date", $date);
        }
        
        $stmt->execute();
        return $stmt;
    }

    // Get records by filters
    public function getByFilters($branch_id = null, $start_date = null, $end_date = null) {
        $query = "SELECT r.*, rid.name as rider_name, u.username as user_name, b.name as branch_name 
                  FROM " . $this->table_name . " r
                  LEFT JOIN riders rid ON r.rider_id = rid.id
                  LEFT JOIN users u ON r.user_id = u.id
                  LEFT JOIN branches b ON r.branch_id = b.id
                  WHERE 1=1";
        
        if($branch_id) {
            $query .= " AND r.branch_id = :branch_id";
        }
        
        if($start_date) {
            $query .= " AND DATE(r.entry_date) >= :start_date";
        }
        
        if($end_date) {
            $query .= " AND DATE(r.entry_date) <= :end_date";
        }
        
        $query .= " ORDER BY r.entry_date DESC";
        
        $stmt = $this->conn->prepare($query);
        
        if($branch_id) {
            $stmt->bindParam(":branch_id", $branch_id);
        }
        
        if($start_date) {
            $stmt->bindParam(":start_date", $start_date);
        }
        
        if($end_date) {
            $stmt->bindParam(":end_date", $end_date);
        }
        
        $stmt->execute();
        return $stmt;
    }

    // Get record by ID
    public function getById($id) {
        $query = "SELECT r.*, rid.name as rider_name, u.username as user_name, b.name as branch_name 
                  FROM " . $this->table_name . " r
                  LEFT JOIN riders rid ON r.rider_id = rid.id
                  LEFT JOIN users u ON r.user_id = u.id
                  LEFT JOIN branches b ON r.branch_id = b.id
                  WHERE r.id = ? LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $id);
        $stmt->execute();
        
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if($row) {
            $this->id = $row['id'];
            $this->rider_id = $row['rider_id'];
            $this->product_name = $row['product_name'];
            $this->quantity = $row['quantity'];
            $this->price = $row['price'];
            $this->location = $row['location'];
            $this->vendor_charges = $row['vendor_charges'];
            $this->payment_mode = $row['payment_mode'];
            $this->notes = $row['notes'];
            $this->user_id = $row['user_id'];
            $this->branch_id = $row['branch_id'];
            $this->entry_date = $row['entry_date'];
            return $row; // Return the full row with related data
        }
        return false;
    }

    // Create record
    public function create() {
        $query = "INSERT INTO " . $this->table_name . " 
                  SET rider_id=:rider_id, product_name=:product_name, quantity=:quantity, 
                      price=:price, location=:location, vendor_charges=:vendor_charges, 
                      payment_mode=:payment_mode, notes=:notes, user_id=:user_id, branch_id=:branch_id";
        
        $stmt = $this->conn->prepare($query);
        
        // Sanitize
        $this->rider_id = htmlspecialchars(strip_tags($this->rider_id));
        $this->product_name = htmlspecialchars(strip_tags($this->product_name));
        $this->quantity = htmlspecialchars(strip_tags($this->quantity));
        $this->price = htmlspecialchars(strip_tags($this->price));
        $this->location = htmlspecialchars(strip_tags($this->location));
        $this->vendor_charges = htmlspecialchars(strip_tags($this->vendor_charges));
        $this->payment_mode = htmlspecialchars(strip_tags($this->payment_mode));
        $this->notes = htmlspecialchars(strip_tags($this->notes));
        $this->user_id = htmlspecialchars(strip_tags($this->user_id));
        $this->branch_id = htmlspecialchars(strip_tags($this->branch_id));
        
        // Bind values
        $stmt->bindParam(":rider_id", $this->rider_id);
        $stmt->bindParam(":product_name", $this->product_name);
        $stmt->bindParam(":quantity", $this->quantity);
        $stmt->bindParam(":price", $this->price);
        $stmt->bindParam(":location", $this->location);
        $stmt->bindParam(":vendor_charges", $this->vendor_charges);
        $stmt->bindParam(":payment_mode", $this->payment_mode);
        $stmt->bindParam(":notes", $this->notes);
        $stmt->bindParam(":user_id", $this->user_id);
        $stmt->bindParam(":branch_id", $this->branch_id);
        
        if($stmt->execute()) {
            return $this->conn->lastInsertId();
        }
        return false;
    }

    // Delete record
    public function delete() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        
        if($stmt->execute()) {
            return true;
        }
        return false;
    }
    
    // Get total records count
    public function getTotalCount() {
        $query = "SELECT COUNT(*) as total FROM " . $this->table_name;
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row['total'];
    }
    
    // Get daily summary
    public function getDailySummary($branch_id = null, $date = null) {
        $query = "SELECT 
                    COUNT(*) as total_transactions,
                    SUM(price) as total_price,
                    SUM(vendor_charges) as total_charges,
                    SUM(price + vendor_charges) as total_revenue
                  FROM " . $this->table_name . " 
                  WHERE 1=1";
        
        if($branch_id) {
            $query .= " AND branch_id = :branch_id";
        }
        
        if($date) {
            $query .= " AND DATE(entry_date) = :date";
        }
        
        $stmt = $this->conn->prepare($query);
        
        if($branch_id) {
            $stmt->bindParam(":branch_id", $branch_id);
        }
        
        if($date) {
            $stmt->bindParam(":date", $date);
        }
        
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
}
?>